define(['angular', 'app', 'localResourceDirectoryService', 'formatter', 'NotificationsConstants'], function (angular, app) {
	"use strict";
	app.service('PatientPreferenceService', function ($http, $filter, $q, localResourceDirectoryService, userSessionService, formatter, NotificationsConstants) {

		var NotificationsService = {},
			notificationTransforms = [],
			isPreferenceStored = null,
			preferencesStoragePromise = $q.defer();

		var initCreatePromise;

		NotificationsService.initializedPromise = null;
		NotificationsService.preferencesStoragePromise = preferencesStoragePromise.promise;

		NotificationsService.dataPointers = {
			notificationPreference: null,
			unreadNotificationsCount: 0,
			emailAddress: null,
			timeZonePreference: null,
			patientPreference: null
		};

		NotificationsService.initializeResources = function () {

			if (NotificationsService.initializedPromise !== null) {
				return NotificationsService.initializedPromise.promise;
			}

			NotificationsService.initializedPromise = $q.defer();

			$q.all([userSessionService.fetch(), localResourceDirectoryService.fetch()])
				.then(function (responses) {
					var user = responses[0];
					NotificationsService.dataPointers.notificationPreference = {
						"firstName": user.patient.firstName,
						"lastName": user.patient.lastName,
						"dateOfBirth": user.patient.dateOfBirth,
						"optInSelected": null
					};
					localResourceDirectoryService.updatePatientIdentifiers(user);
					NotificationsService.initializedPromise.resolve(responses[1]);
				});

			return NotificationsService.initializedPromise.promise;
		};

		var makeNotificationRequest = function (config, linkTitle) {
			var notificationRequestDefer = $q.defer();
			
			var sendRequest = function(config) {
				var defer = $q.defer();
				var notificationTransformKeys = Object.keys(notificationTransforms);

				function transformEntry(item) {
					item.formattedEffectiveDisplayDate = formatter.getFormattedFrontendDateTime(item.effectiveDisplayDate);

					notificationTransforms.forEach(function (transformFn) {
						item = transformFn(item);
					});

					return item;
				}

				if(config.data) {
					delete config.data.link
				}
				
				$http(config).success(function (response, status) {
					if (status !== 204) {
						switch (response['objectType']) {
							case "Notification" :
								response = transformEntry(response);
								break;
							case "Notifications" :
								if (response.notifications) {
									response.notifications.forEach(function (item) {
										item = transformEntry(item);
									});
								} else {
									response.notifications = [];
								}
								break;
						}
						preferencesStoragePromise.resolve();
						isPreferenceStored = true;
						defer.resolve(response);
					} else {
						isPreferenceStored = false;
						defer.reject(response);
					}
				}).error(function (error, code, headers, config) {
					defer.reject(error);
				});
				
				return defer.promise;
			};

			var previousRequestConfig = config;
			var createPreferences = function() {
				//  On initialize, it is possible that this request is going to notification services for a new patient.
				//  The logic is that on a 404, notification services may not have this new patient in the data store.
				//  MyVA Health Journal should see this possibility and redirect to initial settings window by rejecting
				//	the notificationRequestDefer and bubbling error event up to the router
			
				if(initCreatePromise) return initCreatePromise;
				var createDefer = $q.defer();
				initCreatePromise = createDefer.promise;

				return initCreatePromise;
			};

			if (config.url) {
				sendRequest(config).then(function(response){
					notificationRequestDefer.resolve(response);
				});
			} else {
				NotificationsService.initializeResources().then(function (resources) {
					config.url = resources[linkTitle];
					config.suppressErrorMsg = true;

					var getPreferencesPromise = sendRequest(config);

					getPreferencesPromise.then(
						function(response){
							notificationRequestDefer.resolve(response);
						},
						function(response){
							notificationRequestDefer.reject(response);
						});
				});
			}

			return notificationRequestDefer.promise;
		};

		NotificationsService.createNotification = function (item) {
			var requestPromise = makeNotificationRequest({method: "POST", data: item}, "notification");
			requestPromise.then(function (response) {
				return response;
			});
			return requestPromise;
		};

		NotificationsService.getNotificationsPreference = function () {
			var requestPromise = makeNotificationRequest({method: "GET"}, "notification-preference");
			requestPromise.then(function (response) {
				NotificationsService.dataPointers.notificationPreference = response;
			});
			return requestPromise;
		};

		NotificationsService.createNotificationsPreference = function (optInSelected) {
			isPreferenceStored = true;
			var defer = $q.defer();
			NotificationsService.initializeResources().then(function () {
				var updatedPreference = angular.copy(NotificationsService.dataPointers.notificationPreference) || {};
				updatedPreference.optInSelected = optInSelected;
				var requestPromise = makeNotificationRequest({method: "POST", data: updatedPreference}, "notification-preference");
				requestPromise.then(function (response) {
					NotificationsService.dataPointers.notificationPreference = response;
					defer.resolve();
				});
			});
			return defer.promise;
		};

		NotificationsService.updateNotificationsPreference = function (optInSelected) {
			var updatedPreference = angular.copy(NotificationsService.dataPointers.notificationPreference) || {};
			updatedPreference.optInSelected = optInSelected;
			var requestPromise = makeNotificationRequest({method: "PUT", data: updatedPreference}, "notification-preference");
			requestPromise.then(function (response) {
				NotificationsService.dataPointers.notificationPreference = response;
			});
			return requestPromise;
		};

		NotificationsService.getNotificationsPreferenceAndCount = function () {
            var notificationPreference = angular.copy(NotificationsService.dataPointers.notificationPreference) || {};
            var notificationType = (notificationPreference.optInSelected) ?
                                    "FIT_HEART" : "FIT_HEART";

			var requestPromise = makeNotificationRequest({
                method: "GET",
                params: {
                    type: notificationType
                }
            }, "notification-preference-and-count");
			requestPromise.then(function (response) {
				NotificationsService.dataPointers.notificationPreference = response.notificationPreference;
				NotificationsService.dataPointers.unreadNotificationsCount = response.notificationsCount;
			});
			return requestPromise;
		};

		NotificationsService.isPreferenceStored = function () {
			return isPreferenceStored;
		};

		NotificationsService.getPreferencesData = function () {
			var requestPromise = makeNotificationRequest({method: "GET"}, "preferences");
			requestPromise.then(function (response) {
				NotificationsService.dataPointers.emailAddress = response.emailAddress;
				NotificationsService.dataPointers.timeZonePreference = response.timeZone;
				NotificationsService.dataPointers.patientPreference = response;
			});
			return requestPromise;
		};

		NotificationsService.createPreferencesData = function (emailAddressPreference, timeZonePreference) {
			var defer = $q.defer();
			var requestPromise;

			var updatedPreference = angular.copy(NotificationsService.dataPointers.patientPreference) || {};

			NotificationsService.initializeResources().then(function () {

				angular.element.extend(true, updatedPreference, {emailAddress: emailAddressPreference || null, timeZone: timeZonePreference});

				requestPromise = makeNotificationRequest({method: "POST", data: updatedPreference}, "preferences");

				requestPromise.then(function (response) {
					NotificationsService.dataPointers.emailAddress = response.emailAddress;
					NotificationsService.dataPointers.timeZonePreference = response.timeZone;
					NotificationsService.dataPointers.patientPreference = response;
					defer.resolve();
				});
			});
			return defer.promise;
		};

		NotificationsService.updatePreferencesData = function (emailAddressPreference, timeZonePreference) {
			var requestPromise;

			var updatedPreference = angular.copy(NotificationsService.dataPointers.patientPreference) || {};


			angular.element.extend(true, updatedPreference, {emailAddress: emailAddressPreference || null, timeZone: timeZonePreference});
			
			requestPromise = makeNotificationRequest({method: "PUT", data: updatedPreference}, "preferences");

			requestPromise.then(function (response) {
				NotificationsService.dataPointers.emailAddress = response.emailAddress;
				NotificationsService.dataPointers.timeZonePreference = response.timeZone;
				NotificationsService.dataPointers.patientPreference = response;
			});
			return requestPromise;
		};

		return NotificationsService;
	});
});